-- create OLTP write workload package for Oracle

-- random inputs: 
--   randbase: 2..1,000,000
--   randtwomill: 2..2,000,000
--   randfivemill: 2..5,000,000

----------
----------

set echo on
set serveroutput on

create or replace package oltp_write as
	procedure oltp_write_01 (randbase IN int);
	procedure oltp_write_01_check;
	procedure oltp_write_01_undo;
	procedure oltp_write_02 (randbase IN int);
	procedure oltp_write_02_check;
	procedure oltp_write_02_undo;
	procedure oltp_write_03 (randfivemill IN int);
	procedure oltp_write_03_check;
	procedure oltp_write_03_undo;
	procedure oltp_write_04 (randfourmill IN int);
	procedure oltp_write_04_check;
	procedure oltp_write_04_undo;
	procedure oltp_write_05 (randbase IN int);
	procedure oltp_write_05_check;
	procedure oltp_write_05_undo;
end oltp_write;
/

create or replace package body oltp_write as

-- procedure oltp_write_01 (randbase)
-- increment randomly chosen updates.p_int by 1 and save old row in update_int_history
-- changes: updates.p_int, update_int_history
-- might use index on: updates.p_key
-- competes with: oltp_read_select_04, oltp_read_select_05, oltp_read_select_06, oltp_write_06
-- checks atomicity, asolation
-- single table 1 row select and update on an int, 1 row insert

procedure oltp_write_01 (randbase IN int) as
	cursor cursor_oltp_write_01 is
		select updates.p_int
			from updates
			where updates.p_key = randbase
			for update of p_int;
begin
	for row_cursor_oltp_write_01 in cursor_oltp_write_01 loop
		update updates 
			set updates.p_int = updates.p_int + 1
			where current of cursor_oltp_write_01;
		insert into update_int_history values (randbase, row_cursor_oltp_write_01.p_int, sysdate);
	end loop;
	commit work;
exception
   when others then  
	close cursor_oltp_write_01;
	rollback;
	raise_application_error (-20000 + (sqlcode*-1), 'Error in oltp_write_01: ' || sqlerrm);
end oltp_write_01;

--

procedure oltp_write_01_check as
	checkval1	float;
	checkval2	float;
begin
	select sum(updates.p_int) into checkval1 from updates;
	select checkval1 - count(*) into checkval1 from update_int_history;
	select oltp_write_01_check_value.int_sum into checkval2 from oltp_write_01_check_value;

	dbms_output.put_line ('oltp_write_01_check should return two values of 500,000,499,999. The actual values are ' || checkval1 || ' and ' || checkval2);
	if (checkval1 = checkval2 and checkval2 = 500000499999) then
		dbms_output.put_line ('Consistency check passed.');
	else
		dbms_output.put_line ('**CONSISTENCY CHECK FAILURE** in oltp_write_01.');
	end if;
	commit work;
end oltp_write_01_check;

--

procedure oltp_write_01_undo as
begin
	dbms_output.put_line ('Undoing effects of oltp_write_01');
	update updates 
		set updates.p_int = updates.p_key
		where updates.p_int <> updates.p_key;
	delete from update_int_history;
	commit work;
end oltp_write_01_undo;

----------

-- procedure oltp_write_02 (randbase)
-- increment randomly chosen updates.p_signed by 1 and save old row in update_signed_history
-- changes: updates.p_signed, update_signed_history
-- might use index on: updates.p_key
-- competes with oltp_read_select_04, oltp_read_select_05, oltp_read_select_06, oltp_read_join_03
-- checks atomicity, asolation
-- single table 1 row select and update on an int, 1 row insert

procedure oltp_write_02 (randbase IN int) as
	cursor cursor_oltp_write_02 is
		select updates.p_signed
			from updates 
			where updates.p_key = randbase
			for update of p_signed;
begin
	for row_cursor_oltp_write_02 in cursor_oltp_write_02 loop
		update updates
			set updates.p_signed = updates.p_signed + 1
			where current of cursor_oltp_write_02;
		insert into update_signed_history values (randbase, row_cursor_oltp_write_02.p_signed, sysdate);
	end loop;
	commit work;
exception
   when others then  
	close cursor_oltp_write_02;
	rollback;
	raise_application_error (-20000 + (sqlcode*-1), 'Error in oltp_write_02: ' || sqlerrm);
end oltp_write_02;

--

procedure oltp_write_02_check as
	checkval1	float;
	checkval2	int;
begin
	select sum(updates.p_signed) into checkval1 from updates;
	select count(*) into checkval2 from update_signed_history;

	dbms_output.put_line ('oltp_write_02_check should return two identical values (the number of times oltp_write_02 has been run). The actual values are ' || checkval1 || ' and ' || checkval2);
	if (checkval1 = checkval2) then
		dbms_output.put_line ('Consistency check passed.');
	else
		dbms_output.put_line ('**CONSISTENCY CHECK FAILURE** in oltp_write_02.');
	end if;
	commit work;
end oltp_write_02_check;

--

procedure oltp_write_02_undo as
	cursor update_cursor is
		select updates.rowid urowid, update_signed_history.p_signed
		from updates, update_signed_history
		where updates.p_key = update_signed_history.p_key
		and update_signed_history.p_signed in
			(select min(update_signed_history.p_signed)
			from update_signed_history, updates
			where update_signed_history.p_key = updates.p_key
			group by update_signed_history.p_key);
begin
	dbms_output.put_line ('Undoing effects of oltp_write_02');
	for update_row in update_cursor loop
		update updates
			set updates.p_signed = update_row.p_signed
			where updates.rowid = update_row.urowid;
	end loop;
	delete from update_signed_history;
	commit work;
end oltp_write_02_undo;

----------

-- procedure oltp_write_03 (randfivemill)
-- moves a randomly selected row from fivemill to del_history
-- changes: fivemill, del_history
-- might use index on: fivemill.h_key
-- competes with oltp_read_join_03
-- checks atomicity, asolation
-- single row select and insert, single row delete

procedure oltp_write_03 (randfivemill IN int) as
	cursor cursor_oltp_write_03 is
		select * from fivemill 
			where fivemill.h_key = randfivemill
			for update;
begin
	for row_cursor_oltp_write_03 in cursor_oltp_write_03 loop
		insert into del_history
			values (row_cursor_oltp_write_03.h_key, row_cursor_oltp_write_03.h_int, row_cursor_oltp_write_03.h_signed, row_cursor_oltp_write_03.h_float, row_cursor_oltp_write_03.h_double, row_cursor_oltp_write_03.h_decim, row_cursor_oltp_write_03.h_date, row_cursor_oltp_write_03.h_code, row_cursor_oltp_write_03.h_name, row_cursor_oltp_write_03.h_address);
		delete from fivemill where current of cursor_oltp_write_03;
	end loop;
	commit work;
exception
   when others then  
	close cursor_oltp_write_03;
	rollback;
	raise_application_error (-20000 + (sqlcode*-1), 'Error in oltp_write_03: ' || sqlerrm);
end oltp_write_03;

--

procedure oltp_write_03_check as
	checkval1	int;
	checkval2	int;
	checksum	int;
begin
	select count(*) into checkval1 from fivemill;
	select count(*) into checkval2 from del_history;
	checksum := checkval1+checkval2;

	dbms_output.put_line ('oltp_write_03_check should return exactly 5,000,000. The actual value is ' || checksum);
	if (checkval1 + checkval2 = 5000000) then
		dbms_output.put_line ('Consistency check passed.');
	else
		dbms_output.put_line ('**CONSISTENCY CHECK FAILURE** in oltp_write_03.');
	end if;
	commit work;
end oltp_write_03_check;

--

procedure oltp_write_03_undo as
begin
	dbms_output.put_line ('Undoing effects of oltp_write_03');
	insert into fivemill 
		select * from del_history;
	delete from del_history;
	commit work;
end oltp_write_03_undo;

----------

-- procedure oltp_write_04 (randfourmill)
-- modifies between 1 and 2 randomly chosen fourmill.t_name and fourmill.t_address values, increments the update counter oltp_write_04_count_updates.num_updates
-- changes: fourmill.t_name, fourmill.t_address, oltp_write_04_count_updates.num_updates
-- might use index on: fourmill.t_key
-- competes with nothing else
-- checks atomicity
-- single table in range select and update on 2 character fields, 1 int field

procedure oltp_write_04 (randfourmill IN int) as
	cursor cursor_oltp_write_04 is
		select * from fourmill
			where (fourmill.t_key between randfourmill and randfourmill + 500)
			and (fourmill.t_name <> 'NewName')
		for update of t_name, t_address;
	numupdates	int;
begin
	numupdates := 0;
	for row_cursor_oltp_write_04 in cursor_oltp_write_04 loop
		numupdates := numupdates + 1;
		update fourmill
			set fourmill.t_name = 'NewName', fourmill.t_address = 'New Address 320B Lakeside Dr., Foster City, CA'
			where current of cursor_oltp_write_04;
	end loop;
	update oltp_write_04_count_updates
		set oltp_write_04_count_updates.num_updates = oltp_write_04_count_updates.num_updates + numupdates;
	commit work;
exception
   when others then  
	close cursor_oltp_write_04;
	rollback;
	raise_application_error (-20000 + (sqlcode*-1), 'Error in oltp_write_04: ' || sqlerrm);
end oltp_write_04;

--

procedure oltp_write_04_check as
	checkval1	int;
	checkval2	int;
begin
	select count(*) into checkval1 from fourmill
		where fourmill.t_name = 'NewName' and fourmill.t_address = 'New Address 320B Lakeside Dr., Foster City, CA';
	select oltp_write_04_count_updates.num_updates into checkval2 from oltp_write_04_count_updates;

	dbms_output.put_line ('oltp_write_04_check should return two identical values. The actual values are ' || checkval1 || ' and ' || checkval2);
	if (checkval1 = checkval2) then
		dbms_output.put_line ('Consistency check passed.');
	else
		dbms_output.put_line ('**CONSISTENCY CHECK FAILURE** in oltp_write_04.');
	end if;
	commit work;
end oltp_write_04_check;

--

procedure oltp_write_04_undo as
begin
	dbms_output.put_line ('Undoing effects of oltp_write_04');
-- we won't restore the original data, but we will modify these rows to move them to a different index branch
	update fourmill
		set fourmill.t_name = 'OldName'
		where fourmill.t_name = 'NewName' and fourmill.t_address = 'New Address 320B Lakeside Dr., Foster City, CA';
	update oltp_write_04_count_updates
		set num_updates = 0;
	commit work;
end oltp_write_04_undo;

----------

-- procedure oltp_write_05 (randbase)
-- does an update on 10 contiguous rows of updates.p_decim and then rolls the changes back
-- changes: (nothing)
-- might use index on updates.p_key 
-- competes with oltp_write_06, dss_select_05, dss_select_06, dss_select_07, dss_select_08, dss_select_09
-- checks atomicity, consistency, asolation
-- single table range update and rollback

procedure oltp_write_05 (randbase IN int) as
begin
	update updates
		set updates.p_decim = abs(updates.p_decim) - 2000000001
		where updates.p_key between randbase and randbase + 9;
	rollback;
exception
   when NO_DATA_FOUND then
	null;
   when others then
	rollback;
	raise_application_error (-20000 + (sqlcode*-1), 'Error in oltp_write_05: ' || sqlerrm);
end oltp_write_05;
			
--

procedure oltp_write_05_check as
	checkval	int;
begin
	select count(*) into checkval from updates
		where updates.p_decim < -1000000000;
	dbms_output.put_line ('oltp_write_05_check should return exactly 0. The actual value is ' || checkval);
	if (checkval = 0) then
		dbms_output.put_line ('Consistency check passed.');
	else
		dbms_output.put_line ('**CONSISTENCY CHECK FAILURE** in oltp_write_05.');
	end if;
	commit work;
end oltp_write_05_check;

--

procedure oltp_write_05_undo as
begin
	dbms_output.put_line ('Undoing effects of oltp_write_05');
-- procedure as self-undoing
	commit work;
end oltp_write_05_undo;

----------

end oltp_write;
/

commit;
exit;
